#!/usr/bin/env python

"""
Autonomous scheduling functionality.

Copyright (C) 2015, 2016 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from imiptools.text import parse_line
from imiptools.handlers.scheduling.manifest import confirmation_functions, \
                                                   locking_functions, \
                                                   retraction_functions, \
                                                   scheduling_functions, \
                                                   unlocking_functions

# Function application/invocation.

def apply_scheduling_functions(handler):

    """
    Apply the scheduling functions for the current object of the given
    'handler'. This function starts a transaction that should be finalised using
    the 'finish_scheduling' function.

    Return a tuple containing the scheduling decision and any accompanying
    description.
    """

    # First, lock the resources to be used.

    start_scheduling(handler)

    # Obtain the actual scheduling functions with arguments.

    schedulers = get_function_calls(handler.get_scheduling_functions(), scheduling_functions)

    # Then, invoke the scheduling functions.

    response = "ACCEPTED"
    description = None

    for fn, args in schedulers:

        # NOTE: Should signal an error for incorrectly configured resources.

        if not fn:
            return "DECLINED", None

        # Keep evaluating scheduling functions, stopping if one declines or
        # gives a null response, or if one delegates to another resource.

        else:
            result = fn(handler, args)
            result, description = result or ("DECLINED", None)

            # Return a negative result immediately.

            if result in ("DECLINED", "DELEGATED"):
                return result, description

            # Modify the eventual response from acceptance if a countering
            # result is obtained.

            elif response == "ACCEPTED":
                response = result

    return response, description

def confirm_scheduling(handler):

    """
    Confirm scheduling using confirmation functions for the current object of
    the given 'handler'. This function continues a transaction that should be
    finalised using the 'finish_scheduling' function.
    """

    # Obtain the actual confirmation functions with arguments.

    functions = get_function_calls(handler.get_scheduling_functions(), confirmation_functions)
    apply_functions(functions, handler)

def retract_scheduling(handler):

    """
    Retract scheduling using retraction functions for the current object of the
    given 'handler'. This function is a complete transaction in itself.
    """

    # First, lock the resources to be used.

    start_scheduling(handler)
    try:

        # Obtain the actual retraction functions with arguments.

        functions = get_function_calls(handler.get_scheduling_functions(), retraction_functions)
        apply_functions(functions, handler)

    # Finally, unlock the resources.

    finally:
        finish_scheduling(handler)

def start_scheduling(handler):

    """
    Apply locking functions for the given scheduling 'functions' and for the
    current object of the given 'handler'.
    """

    # Obtain functions to lock resources.

    functions = get_function_calls(handler.get_scheduling_functions(), locking_functions)
    apply_functions(functions, handler)

def finish_scheduling(handler):

    """
    Finish scheduling using the given scheduling 'functions' for the current
    object of the given 'handler'.
    """

    # Obtain functions to unlock resources.

    functions = get_function_calls(handler.get_scheduling_functions(), unlocking_functions)
    apply_functions(functions, handler)

def apply_functions(functions, handler):

    """
    Apply the given notification 'functions' for the current object of the given
    'handler'. Where functions are provided more than once, they will be called
    only once for each distinct set of arguments.
    """

    applied = set()

    for fn, args in functions:

        # NOTE: Should signal an error for incorrectly configured resources.

        if not fn or (fn, args) in applied:
            continue

        fn(handler, args)
        applied.add((fn, args))

# Function lookup.

def get_function_calls(lines, registry):

    """
    Parse the given 'lines', returning a list of (function, arguments) tuples,
    with each function being a genuine function object and with the arguments
    being a list of strings.

    Each of the 'lines' should employ the function name and argument strings
    separated by whitespace, with any whitespace inside arguments quoted using
    single or double quotes.

    The given 'registry' indicates the mapping from function names to actual
    functions.
    """

    functions = []

    for line in lines:
        parts = parse_line(line)

        # A sequence of functions is provided for each name.

        l = registry.get(parts[0])
        if l:
            for function in l:
                functions.append((function, tuple(parts[1:])))

    return functions

# vim: tabstop=4 expandtab shiftwidth=4
