#!/usr/bin/env python

"""
Handlers for a person for whom scheduling is performed, inspecting outgoing
messages to obtain scheduling done externally.

Copyright (C) 2014, 2015, 2016 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from imiptools.client import Client
from imiptools.data import get_uri, uri_dict, uri_values
from imiptools.handlers import Handler
from imiptools.handlers.common import CommonEvent

class PersonHandler(CommonEvent, Handler):

    "Handling mechanisms specific to people."

    def set_identity(self, method):

        """
        Set the current user for the current object in the context of the given
        'method'. It is usually set when initialising the handler, using the
        recipient details, but outgoing messages do not reference the recipient
        in this way.
        """

        if self.obj and not self.user:
            from_organiser = method in self.organiser_methods
            if from_organiser:
                self.user = get_uri(self.obj.get_value("ORGANIZER"))

            # Since there may be many attendees in an attendee-provided outgoing
            # message, because counter-proposals can have more than one
            # attendee, the attendee originating from the calendar system is
            # chosen.

            else:
                self.user = self.get_sending_attendee()

    def _add(self):

        "Add a recurrence for the current object."

        if not Client.is_participating(self):
            return False

        # Check for event using UID.

        if not self.have_new_object():
            return False

        # Ignore unknown objects.

        if not self.get_stored_object_version():
            return

        # Record the event as a recurrence of the parent object.

        self.update_recurrenceid()

        # Set the additional occurrence.

        self.store.set_event(self.user, self.uid, self.recurrenceid, self.obj.to_node())

        # Remove any previous cancellations involving this event.

        self.store.remove_cancellation(self.user, self.uid, self.recurrenceid)

        # Update free/busy information.

        self.update_event_in_freebusy()

        return True

    def _record(self, from_organiser=True, counter=False):

        """
        Record details from the current object given a message originating
        from an organiser if 'from_organiser' is set to a true value.
        """

        if not Client.is_participating(self):
            return False

        # Check for a new event, tolerating not-strictly-new events if the
        # attendee is responding.

        if not self.have_new_object(strict=from_organiser):
            return False

        # Update the object.

        if from_organiser:

            # Set the complete event or an additional occurrence.

            self.store.set_event(self.user, self.uid, self.recurrenceid, self.obj.to_node())

            # Remove additional recurrences if handling a complete event.
            # Also remove any previous cancellations involving this event.

            if not self.recurrenceid:
                self.store.remove_recurrences(self.user, self.uid)
                self.store.remove_cancellations(self.user, self.uid)
            else:
                self.store.remove_cancellation(self.user, self.uid, self.recurrenceid)

        else:
            # Occurrences that are still part of a parent object are separated,
            # attendance information transferred, and the free/busy details
            # updated.

            if self.is_newly_separated_occurrence():
                self.make_separate_occurrence(for_organiser=not from_organiser)

            # Obtain valid attendees, merging their attendance with the stored
            # object.

            else:
                attendees = self.require_attendees(from_organiser)
                self.merge_attendance(attendees)

        # Remove any associated request.

        if from_organiser or self.has_indicated_attendance():
            self.store.dequeue_request(self.user, self.uid, self.recurrenceid)
        self.store.remove_counters(self.user, self.uid, self.recurrenceid)

        # Update free/busy information.

        if not counter:
            self.update_event_in_freebusy(from_organiser)

        # For countered proposals, record the offer in the resource's
        # free/busy collection.

        else:
            self.update_event_in_freebusy_offers()

        return True

    def _remove(self):

        """
        Remove details from the current object given a message originating
        from an organiser if 'from_organiser' is set to a true value.
        """

        if not Client.is_participating(self):
            return False

        # Check for event using UID.

        if not self.have_new_object():
            return False

        # Obtain any stored object, using parent object details if a newly-
        # indicated occurrence is referenced.

        obj = self.get_stored_object_version()
        old = not obj and self.get_parent_object() or obj

        if not old:
            return False

        # Only cancel the event completely if all attendees are given.

        attendees = uri_dict(old.get_value_map("ATTENDEE"))
        all_attendees = set(attendees.keys())
        given_attendees = set(uri_values(self.obj.get_values("ATTENDEE")))
        cancel_entire_event = not all_attendees.difference(given_attendees)

        # Update the recipient's record of the organiser's schedule.

        self.remove_freebusy_from_organiser(self.obj.get_value("ORGANIZER"))

        # Otherwise, remove the given attendees and update the event.

        if not cancel_entire_event and obj:
            for attendee in given_attendees:
                if attendees.has_key(attendee):
                    del attendees[attendee]
            obj["ATTENDEE"] = attendees.items()

        # Update the stored object with sequence information.

        if obj:
            obj["SEQUENCE"] = self.obj.get_items("SEQUENCE") or []
            obj["DTSTAMP"] = self.obj.get_items("DTSTAMP") or []

        # Update free/busy information.

        if cancel_entire_event or self.user in given_attendees:
            self.remove_event_from_freebusy()
            self.remove_freebusy_from_attendees(attendees)

        # Set the complete event if not an additional occurrence. For any newly-
        # indicated occurrence, use the received event details.

        self.store.set_event(self.user, self.uid, self.recurrenceid, (obj or self.obj).to_node())

        # Perform any cancellation after recording the latest state of the
        # event.

        if cancel_entire_event:
            self.store.cancel_event(self.user, self.uid, self.recurrenceid)

        # Remove any associated request.

        self.store.dequeue_request(self.user, self.uid, self.recurrenceid)
        self.store.remove_counters(self.user, self.uid, self.recurrenceid)

        return True

    def _declinecounter(self):

        "Remove any counter-proposals for the given event."

        if not Client.is_participating(self):
            return False

        # Check for event using UID.

        if not self.have_new_object():
            return False

        self.remove_counters(uri_values(self.obj.get_values("ATTENDEE")))

class Event(PersonHandler):

    "An event handler."

    def add(self):

        "Record the addition of a recurrence to an event."

        self._add()

    def cancel(self):

        "Remove an event or a recurrence."

        self._remove()

    def counter(self):

        "Record an offer made by a counter-proposal."

        self._record(False, True)

    def declinecounter(self):

        "Expire any offer made by a counter-proposal."

        self._declinecounter()

    def publish(self):

        "Published events are recorded."

        self._record(True)

    def refresh(self):

        "Requests to refresh events do not provide event information."

        pass

    def reply(self):

        "Replies to requests are inspected for attendee information."

        self._record(False)

    def request(self):

        "Record events sent for potential scheduling."

        self._record(True)

# Handler registry.

handlers = [
    ("VEVENT",      Event),
    ]

# vim: tabstop=4 expandtab shiftwidth=4
