package pgtype

import (
	"database/sql/driver"

	"github.com/pkg/errors"
)

// ACLItem is used for PostgreSQL's aclitem data type. A sample aclitem
// might look like this:
//
//	postgres=arwdDxt/postgres
//
// Note, however, that because the user/role name part of an aclitem is
// an identifier, it follows all the usual formatting rules for SQL
// identifiers: if it contains spaces and other special characters,
// it should appear in double-quotes:
//
//	postgres=arwdDxt/"role with spaces"
//
type ACLItem struct {
	String string
	Status Status
}

func (dst *ACLItem) Set(src interface{}) error {
	switch value := src.(type) {
	case string:
		*dst = ACLItem{String: value, Status: Present}
	case *string:
		if value == nil {
			*dst = ACLItem{Status: Null}
		} else {
			*dst = ACLItem{String: *value, Status: Present}
		}
	default:
		if originalSrc, ok := underlyingStringType(src); ok {
			return dst.Set(originalSrc)
		}
		return errors.Errorf("cannot convert %v to ACLItem", value)
	}

	return nil
}

func (dst *ACLItem) Get() interface{} {
	switch dst.Status {
	case Present:
		return dst.String
	case Null:
		return nil
	default:
		return dst.Status
	}
}

func (src *ACLItem) AssignTo(dst interface{}) error {
	switch src.Status {
	case Present:
		switch v := dst.(type) {
		case *string:
			*v = src.String
			return nil
		default:
			if nextDst, retry := GetAssignToDstType(dst); retry {
				return src.AssignTo(nextDst)
			}
		}
	case Null:
		return NullAssignTo(dst)
	}

	return errors.Errorf("cannot decode %#v into %T", src, dst)
}

func (dst *ACLItem) DecodeText(ci *ConnInfo, src []byte) error {
	if src == nil {
		*dst = ACLItem{Status: Null}
		return nil
	}

	*dst = ACLItem{String: string(src), Status: Present}
	return nil
}

func (src *ACLItem) EncodeText(ci *ConnInfo, buf []byte) ([]byte, error) {
	switch src.Status {
	case Null:
		return nil, nil
	case Undefined:
		return nil, errUndefined
	}

	return append(buf, src.String...), nil
}

// Scan implements the database/sql Scanner interface.
func (dst *ACLItem) Scan(src interface{}) error {
	if src == nil {
		*dst = ACLItem{Status: Null}
		return nil
	}

	switch src := src.(type) {
	case string:
		return dst.DecodeText(nil, []byte(src))
	case []byte:
		srcCopy := make([]byte, len(src))
		copy(srcCopy, src)
		return dst.DecodeText(nil, srcCopy)
	}

	return errors.Errorf("cannot scan %T", src)
}

// Value implements the database/sql/driver Valuer interface.
func (src *ACLItem) Value() (driver.Value, error) {
	switch src.Status {
	case Present:
		return src.String, nil
	case Null:
		return nil, nil
	default:
		return nil, errUndefined
	}
}
