#include "avm_args.h"
#include "plugin.h"
#include "configfile.h"
#include "aviplay.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/utsname.h>
#include <ctype.h>


AVM_BEGIN_NAMESPACE;

static bool is_help(const char* str)
{
    return (((toupper(str[0]) == 'H')
	     && (str[1] == 0 || (strcasecmp(str + 1, "ELP") == 0)))
	    || (str[0] == '?' && str[1] == 0));
}

static void split(avm::vector<avm::string>& arr, const char* str)
{
    if (str)
    {
	//printf("SPLIT %s\n", str);
	const char* b;
	while ((b = strchr(str, ':')))
	{
	    if (b > str)
		arr.push_back(avm::string(str, (uint_t)(b - str)));
	    str = b + 1;
	}
	if (str[0])
	    arr.push_back(str);
	for (unsigned i = 0; i < arr.size(); ++i)
	    printf("ARR %d  %s\n", i, arr[i].c_str());
    }
}

static void show_attrs(const CodecInfo& ci, const avm::vector<AttributeInfo>& attr, const char* title)
{
    if (attr.size())
    {
	int val;
	printf("    %s:\n", title);
	for (unsigned i = 0; i < attr.size(); i++)
	{
	    printf("	  %20s", attr[i].GetName());
	    switch (attr[i].GetKind())
	    {
	    case AttributeInfo::Integer:
		PluginGetAttrInt(ci, attr[i].GetName(), &val);
		printf(" %d  default: %d  <%d, %d>", val,
		       attr[i].GetDefault(), attr[i].GetMin(), attr[i].GetMax());
		break;
	    case AttributeInfo::Select:
		PluginGetAttrInt(ci, attr[i].GetName(), &val);
		printf(" %s  default: %s <", attr[i].GetOptions()[val].c_str(),
		       attr[i].GetOptions()[attr[i].GetDefault()].c_str());
		for (int j = attr[i].GetMin(); j < attr[i].GetMax(); j++)
		{
		    if (j != attr[i].GetMin())
			fputs(", ", stdout);
		    fputs(attr[i].GetOptions()[j].c_str(), stdout);
		}
		fputc('>', stdout);
		break;
	    default:
		;
	    }
	    fputc('\n', stdout);
	}
    }
    else
	printf("    No %s\n", title);
}

static void set_codec_defaults(const CodecInfo& ci, const avm::vector<AttributeInfo>& attr)
{
    for (unsigned i = 0; i < attr.size(); i++)
    {
	switch (attr[i].GetKind())
	{
	case AttributeInfo::Integer:
	case AttributeInfo::Select:
	    PluginSetAttrInt(ci, attr[i].GetName(), attr[i].GetDefault());
	default:
	    break;
	}
    }
}

static bool read_bool(const Args::Option& o, const char* arg, const char* par, const char* r)
{
    bool b = true;
    bool rs = false;

    if (par)
    {
	if (!strcasecmp(par, "off") || !strcmp(par, "0")
	    || !strcasecmp(par, "false"))
	{
	    b = false;
	    rs = true;
	}
	else if (!strcasecmp(par, "on") || !strcmp(par, "1")
		 || !strcasecmp(par, "true"))
	{
	    rs = true;
	}
    }

    if (o.is(Args::Option::REGBOOL))
	RegWriteInt(r, o.getLongOption(), b ? 1 : 0);
    else
	o.setBool(b);
    return rs;
}

static int read_double(const Args::Option& o, const char* arg, const char* par, const char* r)
{
    if (!par)
    {
	printf("Option: %s  - missing float value\n", arg);
	return -1;
    }
    double d = atof(par);
    if (!o.isInRange((int32_t)d))
    {
	    printf("Option: %s	- value: %f  out of range <%d, %d>",
		   arg, d, o.getMin(), o.getMax());
	    return -1;
    }
    if (o.is(Args::Option::REGDOUBLE))
	return RegWriteFloat(r, o.getLongOption(), (float)d);

    return o.setDouble(d);
}

static int read_int(const Args::Option& o, const char* arg, const char* par, const char* r)
{
    if (!par)
    {
	printf("Option: %s  - missing integer value\n", arg);
	return -1;
    }
    int v = 0;
    sscanf(par, "%i", &v);
    //printf("READINT %s  %s  %d\n", arg, par, v);
    if (!o.isInRange(v))
    {
	    printf("Option: %s	- value: %d  out of range <%d, %d>",
		   arg, v, o.getMin(), o.getMax());
	    return -1;
    }
    if (o.is(Args::Option::REGINT))
	return RegWriteInt(r, o.getLongOption(), v);
    return o.setInt(v);
}

static int read_string(const Args::Option& o, const char* arg, const char* par, const char* r)
{
    if (!par)
    {
	printf("Option: %s  - missing string value\n", arg);
	return -1;
    }
    if (o.is(Args::Option::REGSTRING))
	return RegWriteString(r, o.getLongOption(), par);

    if (o.setString(strdup(par)))
	return -1;

    return 0;
}

static void show_help(const Args::Option* o, bool prefix)
{
    size_t max = 0;
    avm::vector<avm::string> l;
    for (unsigned i = 0; !o[i].is(Args::Option::NONE); ++i)
    {
	avm::string s;
	if (o[i].is(Args::Option::HELP))
	    s.sprintf("  -h  --help");
	else
	    s.sprintf("  %c%s  %s%s",
		      (o[i].getShortOption()[0] && prefix) ? '-' : ' ',
		      o[i].getShortOption()[0] ? o[i].getShortOption() : " ",
		      (o[i].getLongOption()[0] && prefix) ? "--" : "",
		      o[i].getLongOption()[0] ? o[i].getLongOption() : ""
		      //opt[i].options ? opt[i].options : "",
		      //o[i].help ? o[i].help : ""
		     );
	l.push_back(s);
	size_t len = l.back().size();
	if (max < len)
	    max = len;
    }
    for (unsigned i = 0; !o[i].is(Args::Option::NONE); ++i)
    {
	if (!o[i].getShortOption()[0] && !o[i].getLongOption()[0]
	    && !o[i].is(Args::Option::HELP)
	    && !o[i].is(Args::Option::OPTIONS))
	    continue;

	if (!o[i].is(Args::Option::OPTIONS))
	{
	    fputs(l[i].c_str(), stdout);
	    for (size_t s = l[i].size(); s <= max; s++)
		fputc(' ', stdout);
	    if (o[i].is(Args::Option::HELP))
		fputs("this help message", stdout);
	}

	if (o[i].getString())
	{
	    switch (o[i].getType())
	    {
	    case Args::Option::INT:
	    case Args::Option::REGINT:
		if (o[i].getHelp()[0])
		    printf(o[i].getHelp(), o[i].getInt(),
			   o[i].getMin(), o[i].getMax());
		break;
	    case Args::Option::STRING:
	    case Args::Option::REGSTRING:
	    case Args::Option::SELECTSTRING:
		if (o[i].getHelp()[0])
		    printf(o[i].getHelp(), o[i].getString());
		break;
	    case Args::Option::OPTIONS:
		show_help(o[i].getOptions(), prefix);
		continue;
	    default:
		if (o[i].getHelp()[0])
		    fputs(o[i].getHelp(), stdout);
		break;
	    }
	}
	else if (o[i].getHelp()[0])
	    fputs(o[i].getHelp(), stdout);

	fputs("\n", stdout);
    }
}

static void parse_suboptions(const Args::Option* o, const char* oname, const char* pars, const char* r)
{
    avm::vector<avm::string> arr;
    split(arr, pars);

    if (!arr.size() || is_help(arr[0].c_str()))
    {
	printf("Available options for '%s' (optA=x:optB=...)\n", oname);
	show_help(o, false);
	exit(0);
    }
    for (unsigned i = 0; i < arr.size(); ++i)
    {
	char* par = strchr(&arr[i][0], '=');
	if (par)
	{
	    *par = 0;
	    par++;
	}

	for (unsigned j = 0; !o[j].is(Args::Option::NONE); j++)
	{
	    if ((arr[i] == o[j].getShortOption())
		|| (arr[i] == o[j].getLongOption()))
	    {
		switch(o[j].getType())
		{
		case Args::Option::BOOL:
		    read_bool(o[j], arr[i].c_str(), par, r);
		    break;
		case Args::Option::DOUBLE:
		    read_double(o[j], arr[i].c_str(), par, r);
		    break;
		case Args::Option::INT:
		    read_int(o[j], arr[i].c_str(), par, r);
		    break;
		case Args::Option::STRING:
		    read_string(o[j], arr[i].c_str(), par, r);
		    break;
		default:
		    ;
		}
	    }
	}
    }
}

static void parse_codec(const Args::Option* o, const char* a)
{
    avm::vector<const CodecInfo*> ci;
    CodecInfo::Get(ci, CodecInfo::Video, CodecInfo::Both);
    CodecInfo::Get(ci, CodecInfo::Audio, CodecInfo::Both);

    avm::vector<avm::string> arr;
    split(arr, a);

    if (!arr.size() || arr[0] == "help")
    {
	const char nm[][8] = { "audio", "video" };
	const char nd[][12] = { "", "encoder", "decoder", "de/encoder" };
	fputs("Available codecs:\n"
	      "Idx	Short name  Long name\n", stdout);
	for (unsigned i = 0; i < ci.size(); i++)
	    printf("%3d %15s  %s  (%s %s)\n", i + 1, ci[i]->GetPrivateName(),
		   ci[i]->GetName(), nm[ci[i]->media], nd[ci[i]->direction]);
	exit(0);
    }

    for (unsigned i = 0; i < ci.size(); ++i)
    {
	const char* cname = ci[i]->GetPrivateName();
	if (arr[0] == cname)
	{
	    if (arr[1] == "help")
	    {
		printf("  Options for %s:\n", cname);
		show_attrs(*ci[i], ci[i]->decoder_info, "Decoding Options");
		show_attrs(*ci[i], ci[i]->encoder_info, "Encoding Options");
		exit(0);
	    }
	    else if (arr[1] == "defaults")
	    {
		set_codec_defaults(*ci[i], ci[i]->decoder_info);
		set_codec_defaults(*ci[i], ci[i]->encoder_info);
	    }
	    else
	    {
		for (unsigned j = 1; j < arr.size(); j++)
		{
		    char* p = strchr(&arr[j][0], '=');
		    int val = 0;
		    bool valid = false;
		    if (p)
		    {
			*p++ = 0;
			if (sscanf(p, "%i", &val) > 0)
			    valid = true;
		    }
		    const AttributeInfo* ai = ci[i]->FindAttribute(arr[j].c_str());
		    if (ai)
		    {
			switch (ai->GetKind())
			{
			case AttributeInfo::Integer:
			    if (!valid)
			    {
				printf("  Option %s for %s needs integer value! (given: %s)\n",
				       arr[j].c_str(), cname, p);
				exit(1);
			    }
			    printf("Setting %s = %d\n", arr[j].c_str(), val);
			    PluginSetAttrInt(*ci[i], arr[j].c_str(), val);
			    break;
			case AttributeInfo::Select:
			default:
			    ;
			}
		    }
		    else
		    {
			printf("  Unknown attribute name '%s' for '%s'\n",
			       arr[j].c_str(), cname);
			exit(1);
		    }
		}
	    }
	    if (o)
		o->setString(strdup(cname)); // FIXME - use const ptr
	    break;
	}
    }
    //for (unsigned i = 0; i < arr.size(); i++) printf("ARG %d	 %s\n", i, arr[i].c_str());
}

void Args::ParseCodecInfo(const char* str)
{
    parse_codec(0, str);
}

Args::Args(const Option* _o, int* _argc, char** _argv,
	   const char* _help, const char* rname)
    : opts(_o), argc(_argc), argv(_argv), help(_help), regname(rname)
{
    int sidx = 1;

    for (idx = 1; idx < *argc; ++idx)
    {
	if (argv[idx][0] == '-')
	{
	    int olong = (argv[idx][1] == '-');
	    if (olong && argv[idx][2] == 0) {
		while (++idx < *argc)	   /* copy after '--' */
		    argv[sidx++] = argv[idx];
		break; // end of options
	    }
	    if (findOpt(olong) == 0)
		continue;
	}
	else if (sidx != idx)
	    argv[sidx] = argv[idx];
	sidx++;
    }

    *argc = sidx;
    //for (idx = 1; idx < sidx; ++idx) printf("Left  %d  %s\n", idx, argv[idx]);
}

Args::~Args()
{
}

int Args::findOpt(int olong)
{
    char* arg = argv[idx] + olong + 1;
    char* par = strchr(arg, '=');
    if (par)
	*par++ = 0;
    else if ((idx + 1) < *argc)
	par = argv[++idx];

    avm::vector<const Option*> ol;
    ol.push_back(opts);
    const Option* o = 0;
    while (ol.size() && (!o || !o->is(Option::NONE)))
    {
	o = ol.front();
	ol.pop_front();
	for (; !o->is(Option::NONE); ++o)
	{
	    //printf("OPTION %c   '%s'	%d   '%s' '%s'\n", o->getType(), arg, olong, o->getShortOption(), o->getLongOption());
	    if (o->is(Option::OPTIONS))
	    {
		ol.push_back(o->getOptions());
		continue;
	    }
	    if (o->is(Option::HELP) && (is_help(arg)))
		break;
	    const char* oarg = (olong) ? o->getLongOption() : o->getShortOption();
	    if (oarg && strcmp(arg, oarg) == 0)
		break;
	}
    }

    switch (o->getType())
    {
    default:
    case Option::NONE:
	return -1;
    case Option::BOOL:
    case Option::REGBOOL:
	if (!read_bool(*o, arg, par, regname))
	    idx--; // no argument given
	break;
    case Option::DOUBLE:
    case Option::REGDOUBLE:
	read_double(*o, arg, par, regname);
	break;
    case Option::INT:
    case Option::REGINT:
	read_int(*o, arg, par, regname);
	break;
    case Option::STRING:
    case Option::REGSTRING:
	read_string(*o, arg, par, regname);
	break;
    case Option::CODEC:
	parse_codec(o, par);
	break;
    case Option::SUBOPTIONS:
	parse_suboptions(o->getOptions(), arg, par, regname);
	break;
    case Option::HELP:
	printf("\nUsage: %s %s\n\n", argv[0], help);
	show_help(opts, true);
	exit(0);
    }
    return 0;
}

AVM_END_NAMESPACE;
