/**
 * Pseudo-random number generator routines.
 * 
 * The internal seed is initialized quite randomly so that this module starts
 * a new different pseudo-random sequence each time it is used.
 * 
 * @file
 * @author Umberto Salsi <salsi@icosaedro.it>
 * @version $Date: 2017/05/18 06:38:34 $
 */

#ifndef PRNG_H
#define PRNG_H

#ifdef prng_IMPORT
	#define EXTERN
#else
	#define EXTERN extern
#endif

/**
 * Set the initial seed. This function can be called to force the generator
 * to return the same sequence of random numbers. If not invoked, the module
 * initializes itself using a seed value calculated mixing the current time
 * (seconds and microseconds), total ticks counter and process ID.
 * @param seed
 */
EXTERN void prng_setSeed(unsigned int seed);

/**
 * Returns a pseudo-random integer number in the specified range.
 * @param min Minimum value expected.
 * @param max Maximum value expected.
 * @return Generated pseudo-random number in the range [min,max].
 */
EXTERN int prng_getIntInRange(int min, int max);

/**
 * Returns a pseudo-random number in the range [0.0,1.0[.
 * @return Generated pseudo-random number in the range [0.0,1.0[.
 */
EXTERN double prng_getDouble(void);

/**
 * Returns a pseudo-random number in the range [-1.0,1.0[.
 * @return Generated pseudo-random number in the range [-1.0,1.0[.
 */
EXTERN double prng_getDouble2(void);

/**
 * Fill-in buffer with random bytes.
 * @param buffer Pointer to the buffer.
 * @param buffer_len Length of the buffer.
 */
EXTERN void prng_fill(void *buffer, int buffer_len);

#undef EXTERN
#endif
